// Fetch all tabs and return their id, title, url, and pinned status
async function getAllTabs() {
  const tabs = await browser.tabs.query({});
  return tabs.map(
    ({ id, title = "", url, pinned }) => ({ id, title, url, pinned })
  );
}

// Get a Set of tab IDs from a window object
const getTabIdsFromWindow = window => new Set(window.tabs.map(tab => tab.id));

// Helper function to calculate the score for placing tabs in a window
const getWindowScore = (tabIdsSet, window) => {
  const windowTabIdsSet = getTabIdsFromWindow(window);
  const commonTabIds = tabIdsSet.intersection(windowTabIdsSet);
  return commonTabIds.size / window.tabs.length;
};

// Find the best window for a given tab cluster
const findBestWindow = (windows, tabIdsSet, clusteredWindows) =>
  windows
    .filter(window => !clusteredWindows.has(window.id))
    .map(window =>
      ({ chosenWindow: window, score: getWindowScore(tabIdsSet, window) }))
    .reduce(
      (best, current) => current.score > best.score ? current : best,
      { score: 0, chosenWindow: windows[0] }
    );

// Determine the tabs that should move to the chosen window and new window
const partitionTabs = (tabIdsSet, chosenWindow, score) => {
  const chosenWindowTabIds = getTabIdsFromWindow(chosenWindow);
  return score >= 0.5
    ? {
      tabIdsToMoveToOtherWindow: chosenWindowTabIds.difference(tabIdsSet),
      tabIdsToMoveToChosenWindow: tabIdsSet.difference(chosenWindowTabIds)
    }
    : {
      tabIdsToMoveToOtherWindow: tabIdsSet,
      tabIdsToMoveToChosenWindow: new Set()
    };
};

// Move tabs to a window, skipping closed tabs
const moveTabsToWindow = async (tabIds, windowId) => {
  const validTabIds = [];
  for (const tabId of tabIds) {
    try {
      await browser.tabs.get(tabId);
      validTabIds.push(tabId);
    } catch (e) {
      // Tab was closed, skip it
    }
  }
  if (validTabIds.length) {
    await browser.tabs.move(validTabIds, { windowId, index: -1 });
  }
};

// Move a cluster of tabs to the appropriate window or create/reuse an existing one
const moveTabsCluster = async (tabIds, clusteredWindows) => {
  const tabIdsSet = new Set(tabIds);
  const windows = await browser.windows.getAll({ populate: true });
  const { chosenWindow, score } =
    findBestWindow(windows, tabIdsSet, clusteredWindows);
  const { tabIdsToMoveToOtherWindow, tabIdsToMoveToChosenWindow } =
    partitionTabs(tabIdsSet, chosenWindow, score);

  await moveTabsToWindow(tabIdsToMoveToChosenWindow, chosenWindow.id);

  if (score >= 0.5) clusteredWindows.add(chosenWindow.id);

  if (tabIdsToMoveToOtherWindow.size > 0) {
    // Attempt to reuse an existing non-clustered window
    const nonClusteredWindow =
      windows.find(w => !clusteredWindows.has(w.id) && w.id !== chosenWindow.id);
    let targetWindowId;

    if (nonClusteredWindow) {
      // Reuse the non-clustered window
      targetWindowId = nonClusteredWindow.id;
      await moveTabsToWindow(tabIdsToMoveToOtherWindow, targetWindowId);
    } else {
      // Create a new window with the first tab
      const firstTabId = tabIdsToMoveToOtherWindow.values().next().value;
      const newWindow =
        await browser.windows.create({ tabId: firstTabId, focused: false });
      targetWindowId = newWindow.id;
      tabIdsToMoveToOtherWindow.delete(firstTabId);
      await moveTabsToWindow(tabIdsToMoveToOtherWindow, targetWindowId);
    }

    if (score < 0.5) clusteredWindows.add(targetWindowId);
  }
};

// Reorganize tabs based on the groups returned by the API
async function reorganizeTabs(tabGroups, allTabs) {
  // Create a map of tab IDs to their pinned status
  const pinnedStatus = new Map(allTabs.map(tab => [tab.id, tab.pinned]));
  const clusteredWindows = new Set();

  // Process each group of tabs
  for (const tabIds of tabGroups) {
    const tabIdSet = new Set(tabIds);
    await moveTabsCluster(tabIdSet, clusteredWindows);

    // Repin tabs that were originally pinned
    for (const tabId of tabIdSet) {
      if (pinnedStatus.get(tabId)) {
        await browser.tabs.update(tabId, { pinned: true });
      }
    }
  }
}

export { getAllTabs, reorganizeTabs }
