class ClusterTabsError extends Error {
  constructor(message, statusCode = null, redirectUrl = null) {
    super(message);
    this.name = 'ClusterTabsError';
    this.statusCode = statusCode;
    this.redirectUrl = redirectUrl;
  }
}

const domains = {
  linkedin: "https://linkedin.clustertabs.com",
  github: "https://github.clustertabs.com",
  google: "https://google.clustertabs.com"
};

async function getOAuthCookie(domain) {
  const opts = { name: "_oauth2_proxy", url: domain, storeId: null }
  return await browser.cookies.get(opts);
}

async function sendTabsToAPI(tabs) {
  // Fetch cookies and determine subdomain
  const linkedinCookie = await getOAuthCookie(domains.linkedin);
  const githubCookie = await getOAuthCookie(domains.github);
  const googleCookie = await getOAuthCookie(domains.google);

  const subdomain =
    googleCookie ? "google" : (linkedinCookie ? "linkedin" : (githubCookie ? "github" : null));
  if (!subdomain) {
    throw new ClusterTabsError(
      "Authentication required. Redirecting to login...",
      401,
      "https://clustertabs.com/login"
    );
  }

  const USER_URL = `https://${subdomain}.clustertabs.com/user`;
  const API_URL = `${USER_URL}/api`;

  const response = await fetch(API_URL, {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify(tabs),
    redirect: "manual",
    mode: "cors"
  });

  const locationHeader = response.headers?.get("Location");

  if (!response.ok) {
    switch (response.status) {
      case 302:
      case 401:
      case 403:
        throw new ClusterTabsError(
          "Authentication required. Redirecting to login...",
          response.status,
          locationHeader || "https://clustertabs.com/login"
        );
      case 303:
        throw new ClusterTabsError(
          "Access forbidden. Redirecting to subscription page...",
          response.status,
          locationHeader || USER_URL
        );
      case 500:
        throw new ClusterTabsError(
          "Server error. Please try again later.",
          response.status
        );
      case 503:
        throw new ClusterTabsError(
          "Service is currently unavailable. Please try again later.",
          response.status
        );
      default:
        throw new ClusterTabsError(
          "Unexpected error. Please login and verify your subscription.",
          response.status,
          "https://clustertabs.com/login"
        );
    }
  }
  return response.json();
}

export { sendTabsToAPI, ClusterTabsError };

